<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://platform.neshan.org
 * @since      1.0.0
 *
 * @package    Neshan_Maps
 * @subpackage Neshan_Maps/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Neshan_Maps
 * @subpackage Neshan_Maps/includes
 * @author     Neshan Platform <platform@neshan.org>
 */
class Neshan_Maps {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Neshan_Maps_Loader $loader Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string $plugin_name The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string $version The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'NESHAN_VERSION' ) ) {
			$this->version = NESHAN_VERSION;
		} else {
			$this->version = '1.0.0';
		}

		$this->plugin_name = 'neshan-maps';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Neshan_Maps_Loader. Orchestrates the hooks of the plugin.
	 * - Neshan_Maps_i18n. Defines internationalization functionality.
	 * - Neshan_Maps_Admin. Defines all hooks for the admin area.
	 * - Neshan_Maps_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-neshan-maps-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-neshan-maps-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-neshan-maps-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-neshan-maps-simple-public.php';

		$this->loader = new Neshan_Maps_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Neshan_Maps_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Neshan_Maps_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Neshan_Maps_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'menu' );
		$this->loader->add_action( 'wp_ajax_save_neshan_map', $plugin_admin, 'save' );
		$this->loader->add_action( 'wp_ajax_delete_neshan_map', $plugin_admin, 'delete' );
		$this->loader->add_action( 'admin_notices', $plugin_admin, 'notices' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Neshan_Maps_Simple_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();

		/**
		 * @since   1.0.0
		 */
		add_shortcode( 'neshan-map', [ $this, 'parse_shortcode_simple' ] );

		/**
		 * @since   1.1.0
		 */
		add_shortcode( 'neshan-map-dynamic', [ $this, 'parse_shortcode_dynamic' ] );
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Neshan_Maps_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Parse neshan_map shortcode
	 *
	 * @since     1.0.0
	 *
	 * @param array  $atts
	 * @param null   $content
	 * @param string $tag
	 *
	 * @return string
	 */
	public function parse_shortcode_simple( $atts = [], $content = null, $tag = '' ) {
		$atts = array_change_key_case( (array) $atts, CASE_LOWER );

		if ( ! isset( $atts['id'] ) ) {
			return;
		}

		global $wpdb;

		$id = sanitize_text_field( $atts["id"] );

		$maps = $wpdb->get_results( "SELECT * FROM neshan_maps WHERE id = {$id}" );

		if ( count( $maps ) === 0 ) {
			return;
		}

		$map = json_decode( $maps[0]->data );

		$plugin_public = new Neshan_Maps_Simple_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		return $plugin_public->render( $id, $map, $content );
	}

	/**
	 * Parse neshan_map_dynamic shortcode
	 *
	 * @since     1.1.0
	 *
	 * @param array  $atts
	 * @param null   $content
	 * @param string $tag
	 *
	 * @return string
	 */
	public function parse_shortcode_dynamic( $atts = [], $content = null, $tag = '' ) {
		$atts = array_change_key_case( (array) $atts, CASE_LOWER );

		if ( ! isset( $atts['api_key'] ) || ! isset( $atts['lat'] ) || ! isset( $atts['lng'] ) ) {
			return;
		}

		$wporg_atts = shortcode_atts( [
			'id'      => time(),
			'zoom'    => 15,
			'maptype' => 'neshan',
			'lat'     => null,
			'lng'     => null,
			'api_key' => null,
			'width'   => '100%',
			'height'  => 400
		], $atts, $tag );

		$map          = new stdClass();
		$map->id      = $wporg_atts['id'];
		$map->lat     = $wporg_atts['lat'];
		$map->lng     = $wporg_atts['lng'];
		$map->api_key = $wporg_atts['api_key'];
		$map->zoom    = $wporg_atts['zoom'];
		$map->maptype = $wporg_atts['maptype'];
		$map->width   = $wporg_atts['width'];
		$map->height  = $wporg_atts['height'];

		$plugin_public = new Neshan_Maps_Simple_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		return $plugin_public->render( $map->id, $map, $content );
	}

}
